/**
 * @file    MatchMaking.cs
 * 
 * @date		2024~
 * @copyright	Copyright © Com2uS Platform Corporation. All Right Reserved.
 * @author		jschoi
 * @since		4.24.5.0
 */
using UnityEngine;
using System;
using System.Text;
using System.Collections;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.IO;

/**
 * @defgroup MatchMaking
 * @{
 * \~korean  Hive MatchMaking은 온라인 매치를 위해 플레이어를 연결하고 관련 데이터를 제공합니다. <br/>
 * Hive MatchMaking 기능을 통해 게임에서 구체적인 매치메이킹 로직을 구현하지 않아도 플레이어의 대전 상대를 간단하게 얻을 수 있습니다. <br/><br/>
 * \~english Hive MatchMaking connects players for online matches and provides relevant data. <br/>
 * Hive MatchMaking feature makes it simple to get matches for your players without having to implement specific matchmaking logic in your game. <br/><br/>
 */
 
namespace hive
{
    /**
     *  \~korean
     * Hive MatchMaking은 온라인 매치를 위해 플레이어를 연결하고 관련 데이터를 제공합니다. <br/>
     * Hive MatchMaking 기능을 통해 게임에서 구체적인 매치메이킹 로직을 구현하지 않아도 플레이어의 대전 상대를 간단하게 얻을 수 있습니다. <br/>
     * 플레이어가 비슷한 수준을 가진 상대와 매치되도록 하여 균형잡힌 게임 경험을 제공합니다. <br/>
     *
     *  \~english
     * Hive MatchMaking connects players for online matches and provides relevant data. <br/>
     * Hive MatchMaking feature makes it simple to get matches for your players without having to implement specific matchmaking logic in your game. <br/>
     * It provides a balanced gaming experience by matching players with opponents of similar skill level. <br/>
     *  \~
     * @since   4.24.5.0
     * @ingroup MatchMaking
     */
	public class MatchMaking {

        /**
         *  \~korean
         * @brief MatchMaking API 결과 및 데이터 통지<br/>
         *
         * @param result                API 호출 결과
         * @param matchMakingData       매치 요청 상태 및 매치 정보
         *
         *  \~english
         * @brief MatchMaking data result callback<br/>
         *
         * @param result                Result of API call
         * @param matchMakingData       Requested match status and match information
         *  
         *  \~
         * @see #requestMatchMaking(String, int, String, onMatchMakingData)
         *
         * @ingroup MatchMaking
         */
		public delegate	void onMatchMakingData(ResultAPI result, MatchMakingData matchMakingData);

        /**
         *  \~korean
         * @brief MatchMaking API 결과 통지<br/>
         *
         * @param result        API 호출 결과
         *
         *  \~english
         * @brief MatchMaking result callback<br/>
         *
         * @param result        Result of API call
         *  
         *  \~
         * @see #deleteRequesting(String, onMatchMakingData)
         *
         * @ingroup MatchMaking
         */
		public delegate void onMatchMakingResult(ResultAPI result);

        /**
         *  \~korean
         * @brief MatchMaking 그룹 매칭 API 결과 및 데이터 통지<br/>
         *
         * @param result                    API 호출 결과
         * @param matchMakingGroupData      매치 요청 상태 및 그룹 매치 정보
         *
         *  \~english
         * @brief MatchMaking group match data result callback<br/>
         *
         * @param result                Result of API call
         * @param matchMakingData       Requested group match status and group match information
         *  
         *  \~
         * @see #requestGroupMatching(String, int, String, onMatchMakingData)
         *
         * @ingroup MatchMaking
         */
		public delegate	void onMatchMakingGroupData(ResultAPI result, MatchMakingGroupData matchMakingGroupData);


 		/**
         *  \~korean
         * 매칭 요청<br/>
         *
         * 매칭 요청 시 사용할 점수(point)를 입력할 수 있습니다.<br/>
         * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 입력할 수 있습니다.<br/>
         * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
         *
         *  \~english
         * Matching Request<br/>
         *
         * You can enter the score(point) to be used when requesting matching.<br/>
         * You can also enter additional information(nickname, level, country, etc.)(extraData) to be used for matching.<br/>
         * ExtraData is delivered as part of the matching results.<br/>
         *  \~
         * @since   4.24.5.0
         * @ingroup MatchMaking
         */
        public static void requestMatchMaking(int matchId, int point, string extraData, onMatchMakingData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "requestMatchMaking", listener);
            jsonParam.AddField ("matchId", matchId);
            jsonParam.AddField ("point", point);
            jsonParam.AddField ("extraData", extraData);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 매칭 상태 확인<br/>
         *
         * 요청한 매칭 상태를 확인할 수 있습니다.<br/>
         *
         *  \~english
         * Check Matching Status<br/>
         *
         * You can check the status of your requested matching.<br/>
         *  \~
         * @since   4.24.5.0
         * @ingroup MatchMaking
         */
        public static void getRequestingStatus(int matchId, onMatchMakingData listener) {
            
            JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "getRequestingStatus", listener);
            
            jsonParam.AddField("matchId", matchId);

            HIVEUnityPlugin.callNative (jsonParam);
        }

        /**
         *  \~korean
         * 매칭 요청 삭제<br/>
         *
         * 요청한 매칭을 삭제합니다.<br/>
         *
         *  \~english
         * Delete Matching Request<br/>
         *
         * Deletes the requested match.<br/>
         *  \~
         * @since   4.24.5.0
         * @ingroup MatchMaking
         */
        public static void deleteRequesting(int matchId, onMatchMakingResult listener) {
            
            JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "deleteRequesting", listener);

            jsonParam.AddField("matchId", matchId);

            HIVEUnityPlugin.callNative (jsonParam);
        }

        /**
         *  \~korean
         * 그룹 생성 요청<br/>
         *
         * 매칭 요청 시 사용할 점수(point)를 입력할 수 있습니다.<br/>
         * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 입력할 수 있습니다.<br/>
         * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
         *
         *  \~english
         * Create group<br/>
         *
         * You can enter the score(point) to be used when requesting matching.<br/>
         * You can also enter additional information(nickname, level, country, etc.)(extraData) to be used for matching.<br/>
         * ExtraData is delivered as part of the matching results.<br/>
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void createGroup(int matchId, int point, string extraData, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "createGroup", listener);
            jsonParam.AddField ("matchId", matchId);
            jsonParam.AddField ("point", point);
            jsonParam.AddField ("extraData", extraData);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 참여<br/>
         *
         * 매칭 요청 시 사용할 점수(point)를 입력할 수 있습니다.<br/>
         * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 입력할 수 있습니다.<br/>
         * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
         *
         *  \~english
         * Join group<br/>
         *
         * You can enter the score(point) to be used when requesting matching.<br/>
         * You can also enter additional information(nickname, level, country, etc.)(extraData) to be used for matching.<br/>
         * ExtraData is delivered as part of the matching results.<br/>
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void joinGroup(int matchId, string groupCode, int point, string extraData, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "joinGroup", listener);
            jsonParam.AddField ("matchId", matchId);
            jsonParam.AddField ("groupCode", groupCode);
            jsonParam.AddField ("point", point);
            jsonParam.AddField ("extraData", extraData);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 탈퇴<br/>
         *
         *  \~english
         * Leave group<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void leaveGroup(int matchId, onMatchMakingResult listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "leaveGroup", listener);
            jsonParam.AddField ("matchId", matchId);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹원 추방<br/>
         *
         *  \~english
         * Kick user from a group<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void kickGroupUser(int matchId, long targetPlayerId, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "kickGroupUser", listener);
            jsonParam.AddField ("matchId", matchId);
            jsonParam.AddField ("targetPlayerId", targetPlayerId);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 정보 조회 (그룹내 유저 기준)<br/>
         *
         *  \~english
         * Search for group information based on users within the group.<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void getGroupInfoByUser(int matchId, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "getGroupInfoByUser", listener);
            jsonParam.AddField ("matchId", matchId);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 정보 조회 (그룹 코드 기준)<br/>
         *
         *  \~english
         * Search for group information based on group code.<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void getGroupInfoByGroupCode(string groupCode, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "getGroupInfoByGroupCode", listener);
            jsonParam.AddField ("groupCode", groupCode);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 멤버 정보 수정<br/>
         *
         * 매칭 요청 시 사용할 점수(point)를 변경할 수 있습니다.<br/>
         * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 변경할 수 있습니다.<br/>
         * ready 상태를 변경할 수 있습니다.
         * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
         *
         * You can change the points to use when requesting a match.<br/>
         * You can also change the additional information (nickname, level, country, etc.) (extraData) to be used for matching.<br/>
         * You can change the "ready" status.
         * ExtraData is delivered as part of the matching results.<br/>
         *
         *  \~english
         * Update group user information.<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void updateGroupUser(int matchId, bool ready, int point, string extraData, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "updateGroupUser", listener);
            jsonParam.AddField ("matchId", matchId);
            jsonParam.AddField ("ready", ready);
            jsonParam.AddField ("point", point);
            jsonParam.AddField ("extraData", extraData);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 매칭 요청<br/>
         *
         *
         *  \~english
         * Group Matching Request<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void requestGroupMatching(int matchId, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "requestGroupMatching", listener);
            jsonParam.AddField ("matchId", matchId);

			HIVEUnityPlugin.callNative (jsonParam);
		}

        /**
         *  \~korean
         * 그룹 매칭 요청 삭제<br/>
         *
         *
         *  \~english
         * Delete Group Matching Request<br/>
         *
         *  \~
         * @since   4.25.0.0
         * @ingroup MatchMaking
         */
        public static void deleteGroupMatching(int matchId, onMatchMakingGroupData listener) {

			JSONObject jsonParam = HIVEUnityPlugin.createParam("MatchMaking", "deleteGroupMatching", listener);
            jsonParam.AddField ("matchId", matchId);

			HIVEUnityPlugin.callNative (jsonParam);
		}


		// \internal
		// \~korean Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
		// \~english Plug-in internal code to handle requests invoked from the native code.
		public static void executeEngine(JSONObject resJsonObject) {

			String methodName = null;
			resJsonObject.GetField (ref methodName, "method");

			int handlerId = -1;
			resJsonObject.GetField (ref handlerId, "handler");
			object handler = (object)HIVEUnityPlugin.popHandler (handlerId);

			if (handler == null) return;

            if ("requestMatchMaking".Equals (methodName)) {
                onMatchMakingData listener = (onMatchMakingData)handler;
                JSONObject data = resJsonObject.GetField("matchMakingData");
                MatchMakingData matchMakingData = new MatchMakingData(data);
                listener (new ResultAPI(resJsonObject.GetField("resultAPI")), matchMakingData);
            }
            else if ("getRequestingStatus".Equals (methodName)) {
                onMatchMakingData listener = (onMatchMakingData)handler;
                JSONObject data = resJsonObject.GetField("matchMakingData");
                MatchMakingData matchMakingData = new MatchMakingData(data);
                listener (new ResultAPI(resJsonObject.GetField("resultAPI")), matchMakingData);
            }
            else if ("deleteRequesting".Equals (methodName)) {
                onMatchMakingResult listener = (onMatchMakingResult)handler;              
                listener (new ResultAPI(resJsonObject.GetField("resultAPI")));
            }
            else if ("createGroup".Equals (methodName) 
                || "joinGroup".Equals (methodName)
                || "kickGroupUser".Equals (methodName)
                || "getGroupInfoByUser".Equals (methodName)
                || "getGroupInfoByGroupCode".Equals (methodName)
                || "updateGroupUser".Equals (methodName)
                || "requestGroupMatching".Equals (methodName)
                || "deleteGroupMatching".Equals (methodName)) {
                onMatchMakingGroupData listener = (onMatchMakingGroupData)handler;
                JSONObject data = resJsonObject.GetField("matchMakingGroupData");
                MatchMakingGroupData matchMakingGroupData = new MatchMakingGroupData(data);
                listener (new ResultAPI(resJsonObject.GetField("resultAPI")), matchMakingGroupData);
            }
            else if ("leaveGroup".Equals (methodName)) {
                onMatchMakingResult listener = (onMatchMakingResult)handler;              
                listener (new ResultAPI(resJsonObject.GetField("resultAPI")));
            }
		}

        /**
         * \~korean  Hive 매치 메이킹 서버에서 반환한 매치 정보
         * 
         * \~english Match information returned by the Hive match making server
         * \~
         * @ingroup MatchMaking
         */
        public class MatchMakingData {

            public long requestPlayerId = 0L;
            public int requestGameIndex = 0;
            public int requestMatchId = 0;

            public string requestStatus = "";
            public string requestTimeUtc = "";
            public int requestPoint = 0;
            public string requestExtraData = "";

            public string matchingStatus = "";
            public string matchingId = "";
            public string matchingType = "unknown";

            public List<MatchingResultPlayerInfo> matchingPlayerInfoList = null;
            public List<MatchingResultTeamInfo> matchingTeamInfoList = null;

            public MatchMakingData() {}

            public MatchMakingData(JSONObject resJsonParam) {

                if(resJsonParam == null)
                    return;

                resJsonParam.GetField(ref this.requestPlayerId, "requestPlayerId");
                resJsonParam.GetField(ref this.requestGameIndex, "requestGameIndex");
                resJsonParam.GetField(ref this.requestMatchId, "requestMatchId");

                resJsonParam.GetField(ref this.requestStatus, "requestStatus");
                resJsonParam.GetField(ref this.requestTimeUtc, "requestTimeUtc");
                resJsonParam.GetField(ref this.requestPoint, "requestPoint");
                resJsonParam.GetField(ref this.requestExtraData, "requestExtraData");

                resJsonParam.GetField(ref this.matchingStatus, "matchingStatus");
                resJsonParam.GetField(ref this.matchingId, "matchingId");
                resJsonParam.GetField(ref this.matchingType, "matchingType");

                if(resJsonParam.HasField("matchingPlayerInfoList")) {
                    JSONObject jsonArray = resJsonParam.GetField("matchingPlayerInfoList");
                    if (jsonArray != null && jsonArray.count > 0) {
                        List<JSONObject> jsonList = jsonArray.list;
                        matchingPlayerInfoList = new List<MatchingResultPlayerInfo>();
                        foreach (JSONObject jsonItem in jsonList) {
                            matchingPlayerInfoList.Add(new MatchingResultPlayerInfo(jsonItem));
                        }
                    }
                } else if(resJsonParam.HasField("matchingTeamInfoList")) {

                    JSONObject jsonArray = resJsonParam.GetField("matchingTeamInfoList");
                    if (jsonArray != null && jsonArray.count > 0) {
                        List<JSONObject> jsonList = jsonArray.list;
                        matchingTeamInfoList = new List<MatchingResultTeamInfo>();
                        foreach (JSONObject jsonItem in jsonList) {
                            matchingTeamInfoList.Add(new MatchingResultTeamInfo(jsonItem));
                        }
                    }
                }
            }

            public JSONObject toJson() {
                JSONObject jsonObject = new JSONObject();

                jsonObject.AddField("requestPlayerId", requestPlayerId);
                jsonObject.AddField("requestGameIndex", requestGameIndex);
                jsonObject.AddField("requestMatchId", requestMatchId);
                
                jsonObject.AddField("requestStatus", requestStatus);
                jsonObject.AddField("requestTimeUtc", requestTimeUtc);
                jsonObject.AddField("requestPoint", requestPoint);
                jsonObject.AddField("requestExtraData", requestExtraData);
                
                jsonObject.AddField("matchingStatus", matchingStatus);
                jsonObject.AddField("matchingId", matchingId);
                jsonObject.AddField("matchingType", matchingType);

                if(matchingPlayerInfoList != null)
                    jsonObject.AddField("matchingPlayerInfoList", matchingPlayerInfoList.ToString());
                
                if(matchingTeamInfoList != null)
                    jsonObject.AddField("matchingTeamInfoList", matchingTeamInfoList.ToString());

                return jsonObject;
            }

            public String toString() {

				StringBuilder sb = new StringBuilder();

				sb.Append("MatchMakingData { requestPlayerId = ");
				sb.Append(this.requestPlayerId);
				sb.Append(", requestGameIndex = ");
				sb.Append(this.requestGameIndex);
				sb.Append(", requestMatchId = ");
				sb.Append(this.requestMatchId);
				sb.Append(", requestStatus = ");
				sb.Append(this.requestStatus);
				sb.Append(", requestTimeUtc = ");
				sb.Append(this.requestTimeUtc);
                sb.Append(", requestPoint = ");
				sb.Append(this.requestPoint);
                sb.Append(", requestExtraData = ");
				sb.Append(this.requestExtraData);
                sb.Append(", matchingStatus = ");
				sb.Append(this.matchingStatus);
                sb.Append(", matchingId = ");
				sb.Append(this.matchingId);
                sb.Append(", matchingType = ");
				sb.Append(this.matchingType);

				sb.Append(", matchingPlayerInfoList = { ");
                if(matchingPlayerInfoList != null) {
                    for(int i = 0; i < matchingPlayerInfoList.Count; i++) {
                         sb.Append(matchingPlayerInfoList[i].toString());
                         if(i < matchingPlayerInfoList.Count - 1) sb.Append(",");
                    }
                }

				sb.Append(" }, matchingTeamInfoList = { ");
                if(matchingTeamInfoList != null) {
                    for(int i = 0; i < matchingTeamInfoList.Count; i++) {
                         sb.Append(matchingTeamInfoList[i].toString());
                         if(i < matchingTeamInfoList.Count - 1) sb.Append(",");
                    }
                }
				sb.Append(" }");

				sb.Append(" }\n");

				return sb.ToString();
			}
        }

        /**
         * \~korean  Hive 매치 메이킹 서버에서 반환한 그룹 매치 정보
         * 
         * \~english Group Match information returned by the Hive match making server
         * \~
         * @ingroup MatchMaking
         */
        public class MatchMakingGroupData {

            public string groupCode = "";
            public long ownerPlayerId = 0L;

            public List<MatchingMemberInfo> memberInfoList = null;
            
            public int requestGameIndex = 0;
            public int requestMatchId = 0;

            public string requestStatus = "";
            public string requestTimeUtc = "";

            public string matchingStatus = "";
            public string matchingId = "";
            public string matchingType = "unknown";

            public List<MatchingResultTeamInfo> matchingTeamInfoList = null;

            public MatchMakingGroupData() {}

            public MatchMakingGroupData(JSONObject resJsonParam) {

                if(resJsonParam == null)
                    return;

                resJsonParam.GetField(ref this.groupCode, "groupCode");
                resJsonParam.GetField(ref this.ownerPlayerId, "ownerPlayerId");
                

                resJsonParam.GetField(ref this.requestGameIndex, "requestGameIndex");
                resJsonParam.GetField(ref this.requestMatchId, "requestMatchId");

                resJsonParam.GetField(ref this.requestStatus, "requestStatus");
                resJsonParam.GetField(ref this.requestTimeUtc, "requestTimeUtc");

                resJsonParam.GetField(ref this.matchingStatus, "matchingStatus");
                resJsonParam.GetField(ref this.matchingId, "matchingId");
                resJsonParam.GetField(ref this.matchingType, "matchingType");

                if(resJsonParam.HasField("memberInfoList")) {
                    JSONObject jsonArray = resJsonParam.GetField("memberInfoList");
                    if (jsonArray != null && jsonArray.count > 0) {
                        List<JSONObject> jsonList = jsonArray.list;
                        memberInfoList = new List<MatchingMemberInfo>();
                        foreach (JSONObject jsonItem in jsonList) {
                            memberInfoList.Add(new MatchingMemberInfo(jsonItem));
                        }
                    }
                }

                if(resJsonParam.HasField("matchingTeamInfoList")) {

                    JSONObject jsonArray = resJsonParam.GetField("matchingTeamInfoList");
                    if (jsonArray != null && jsonArray.count > 0) {
                        List<JSONObject> jsonList = jsonArray.list;
                        matchingTeamInfoList = new List<MatchingResultTeamInfo>();
                        foreach (JSONObject jsonItem in jsonList) {
                            matchingTeamInfoList.Add(new MatchingResultTeamInfo(jsonItem));
                        }
                    }
                }
            }

            public JSONObject toJson() {
                JSONObject jsonObject = new JSONObject();

                jsonObject.AddField("groupCode", groupCode);
                jsonObject.AddField("ownerPlayerId", ownerPlayerId);

                if(memberInfoList != null)
                    jsonObject.AddField("memberInfoList", memberInfoList.ToString());
                
                jsonObject.AddField("requestGameIndex", requestGameIndex);
                jsonObject.AddField("requestMatchId", requestMatchId);
                jsonObject.AddField("requestStatus", requestStatus);
                jsonObject.AddField("requestTimeUtc", requestTimeUtc);

                jsonObject.AddField("matchingStatus", matchingStatus);
                jsonObject.AddField("matchingId", matchingId);
                jsonObject.AddField("matchingType", matchingType);

                if(matchingTeamInfoList != null)
                    jsonObject.AddField("matchingTeamInfoList", matchingTeamInfoList.ToString());

                return jsonObject;
            }

            public String toString() {

				StringBuilder sb = new StringBuilder();

				sb.Append("MatchMakingGroupData { groupCode = ");
				sb.Append(this.groupCode);
                sb.Append(", ownerPlayerId = ");
				sb.Append(this.ownerPlayerId);

				sb.Append(", requestGameIndex = ");
				sb.Append(this.requestGameIndex);
				sb.Append(", requestMatchId = ");
				sb.Append(this.requestMatchId);
				sb.Append(", requestStatus = ");
				sb.Append(this.requestStatus);
				sb.Append(", requestTimeUtc = ");
				sb.Append(this.requestTimeUtc);

                sb.Append(", matchingStatus = ");
				sb.Append(this.matchingStatus);
                sb.Append(", matchingId = ");
				sb.Append(this.matchingId);
                sb.Append(", matchingType = ");
				sb.Append(this.matchingType);

				sb.Append(", memberInfoList = { ");
                if(memberInfoList != null) {
                    for(int i = 0; i < memberInfoList.Count; i++) {
                        sb.Append(memberInfoList[i].toString());
                        if(i < memberInfoList.Count - 1) sb.Append(",");
                    }
                }
				sb.Append(" }");

				sb.Append(", matchingTeamInfoList = { ");
                if(matchingTeamInfoList != null) {
                    for(int i = 0; i < matchingTeamInfoList.Count; i++) {
                        sb.Append(matchingTeamInfoList[i].toString());
                        if(i < matchingTeamInfoList.Count - 1) sb.Append(",");
                    }
                }
				sb.Append(" }");

				sb.Append(" }\n");

				return sb.ToString();
			}
        }

        public class MatchingMemberInfo {

			public long playerId = 0L;
			public bool ready = false;
            public int point = 0;
			public string extraData = "";

			public MatchingMemberInfo() {}

			public MatchingMemberInfo(long playerId, bool ready, int point, string extraData) {

				this.playerId = playerId;
				this.ready = ready;
                this.point = point;
				this.extraData = extraData;
			}

			public MatchingMemberInfo(JSONObject jsonParam) {

				if (jsonParam == null || jsonParam.count <= 0) return;

				jsonParam.GetField (ref this.playerId, "playerId");
				jsonParam.GetField (ref this.ready, "ready");
                jsonParam.GetField (ref this.point, "point");
				jsonParam.GetField (ref this.extraData, "extraData");
			}

			public String toString() {

				StringBuilder sb = new StringBuilder();

				sb.Append(" { playerId = ");
				sb.Append(this.playerId);
				sb.Append(", ready = ");
				sb.Append(this.ready);
                sb.Append(", point = ");
				sb.Append(this.point);
				sb.Append(", extraData = ");
				sb.Append(this.extraData);
				sb.Append(" }\n");

				return sb.ToString();
			}
		}

        public class MatchingResultPlayerInfo {

			public long playerId = 0L;
			public int point = 0;
			public string extraData = "";

			public MatchingResultPlayerInfo() {}

			public MatchingResultPlayerInfo(long playerId, int point, string extraData) {

				this.playerId = playerId;
				this.point = point;
				this.extraData = extraData;
			}

			public MatchingResultPlayerInfo(JSONObject jsonParam) {

				if (jsonParam == null || jsonParam.count <= 0) return;

				jsonParam.GetField (ref this.playerId, "playerId");
				jsonParam.GetField (ref this.point, "point");
				jsonParam.GetField (ref this.extraData, "extraData");
			}

			public String toString() {

				StringBuilder sb = new StringBuilder();

				sb.Append(" { playerId = ");
				sb.Append(this.playerId);
				sb.Append(", point = ");
				sb.Append(this.point);
				sb.Append(", extraData = ");
				sb.Append(this.extraData);
				sb.Append(" }\n");

				return sb.ToString();
			}
		}

        public class MatchingResultTeamInfo {

			public int teamIndex = -1;
			public List<MatchingResultPlayerInfo> playerInfos = null;

			public MatchingResultTeamInfo() {}

			public MatchingResultTeamInfo(int teamIndex, List<MatchingResultPlayerInfo> playerInfos) {

				this.teamIndex = teamIndex;
				this.playerInfos = playerInfos;
			}

			public MatchingResultTeamInfo(JSONObject jsonParam) {

				if (jsonParam == null || jsonParam.count <= 0) return;

				jsonParam.GetField (ref this.teamIndex, "teamIndex");

                JSONObject jsonArray = jsonParam.GetField("playerInfos");
                if (jsonArray != null && jsonArray.count > 0) {
                    List<JSONObject> jsonList = jsonArray.list;
                    playerInfos = new List<MatchingResultPlayerInfo>();
                    foreach (JSONObject jsonItem in jsonList) {
                        playerInfos.Add(new MatchingResultPlayerInfo(jsonItem));
                    }
                }
			}

			public String toString() {

				StringBuilder sb = new StringBuilder();

				sb.Append(" { teamIndex = ");
				sb.Append(this.teamIndex);
				
                sb.Append(", playerInfos = { ");
                if(playerInfos != null) {
                    for(int i = 0; i < playerInfos.Count; i++) {
                         sb.Append(playerInfos[i].toString());
                         if(i < playerInfos.Count - 1) sb.Append(",");
                    }
                }
                sb.Append(" }");

				sb.Append(" }\n");

				return sb.ToString();
			}
		}
	}
}


/** @} */



